#include "Config.h"
/******************************************************************************
 * GUI.cpp - Gestion de l'interface graphique                                 *
 ******************************************************************************
 * Projet       : Archive(R)                                                  *
 * Auteur       : Arnaud Storq (http://norecess.planet-d.net)                 *
 ******************************************************************************/
#include "GUI.h"

// ---------------------------------------------------------------------------- DEFINES
// La taille fixe de notre fenetre !
// Libre  vous de modifier ces valeurs pour agrandir la fenetre de l'outil..
#define FENETRE_LARGEUR 512
#define FENETRE_HAUTEUR 320

// ---------------------------------------------------------------------------- IDENTIFIANTS
enum
{
    // Identifiant de l'application. Nous aurions pu nous en passer puisque nous
    // n'avons qu'une seule fenetre dans l'application, mais cela facilite
    // l'eventuelle evolutivit de l'outil.
    ID_APPLICATION          = wxID_HIGHEST+1,
   
    // Identifiants qui nous serviront  lier le menu avec nos methodes.
    ID_OUVRIR               = wxID_HIGHEST+101,
    ID_SAUVEGARDER          = wxID_HIGHEST+102,
    ID_AJOUTER              = wxID_HIGHEST+103,
    ID_SUPPRIMER            = wxID_HIGHEST+104,
    ID_EXTRAIRE             = wxID_HIGHEST+105,
    ID_EXTRAIRE_TOUT        = wxID_HIGHEST+106,
    ID_A_PROPOS             = wxID_HIGHEST+107,
    ID_QUITTER              = wxID_HIGHEST+108,
   
    // Identifiant du controle liste.
    ID_LISTE                = wxID_HIGHEST+201
};

// ---------------------------------------------------------------------------- TABLE D'EVENEMENTS
// Cela permet de lier les evenements d'un controle  une fonction de notre classe.
BEGIN_EVENT_TABLE(GUI, wxFrame)
    EVT_MENU(ID_OUVRIR, GUI::archiveOuvrir)
    EVT_MENU(ID_SAUVEGARDER, GUI::archiveSauvegarder)
    EVT_MENU(ID_AJOUTER, GUI::fichierAjouter)
    EVT_MENU(ID_SUPPRIMER, GUI::fichierSupprimer)
    EVT_MENU(ID_EXTRAIRE, GUI::fichierExtraire)
    EVT_MENU(ID_EXTRAIRE_TOUT, GUI::fichierExtraireTout)
    EVT_MENU(ID_A_PROPOS, GUI::archiveurAPropos)
    EVT_MENU(ID_QUITTER, GUI::archiveurQuitter)
END_EVENT_TABLE()

// ---------------------------------------------------------------------------- CONSTRUCTEUR
GUI::GUI()
{
    // Initialiser la fenetre
    initialiserFenetre();
    
    // Initialiser le menu
    initialiserMenu();
    
    // Initialiser le controle liste
    initialiserListe();
    
    // Cre une barre de statut, avec une seule case.
    // Amusez-vous  rajouter une autre case et parcourez la documentation 
    // pour la manipulation de la Status Bar. C'est trs simple !
    CreateStatusBar(1);
}

// ---------------------------------------------------------------------------- INITIALISER FENETRE
void GUI::initialiserFenetre()
{
    // Crer la fenetre de l'application
    Create((wxWindow *) NULL, // la fenetre parente est le bureau (NULL)
            ID_APPLICATION, // identifiant de la fenetre
            "Archive(R) - Exemple d'outil realis avec Dev-C++ et wxWindows", // titre
            wxPoint(-1, -1), // une position de -1;-1 laisse le systme placer la fenetre comme bon lui semble
            wxSize(FENETRE_LARGEUR, FENETRE_HAUTEUR), // la taille de la fenetre est fixe
            wxDEFAULT_FRAME_STYLE & ~wxRESIZE_BORDER & ~wxRESIZE_BOX & ~wxMAXIMIZE_BOX, // indique une fenetre de type normale mais modification de taille 
            "Archive(R)" // nom d'instance
            );
    
    // Centrer la fenetre par rapport  l'ecran
    Centre();   
    
    // Attacher une icone a la fenetre
    wxIcon icon;
    icon.LoadFile( "Archive(R).ico", wxBITMAP_TYPE_ICO);
    SetIcon(icon);
}

// ---------------------------------------------------------------------------- INITIALISER MENU
void GUI::initialiserMenu()
{
    wxMenuBar *MenuBar = new wxMenuBar;

    // Menu Fichier
    {
        wxMenu *menuFichier = new wxMenu;
    
        menuFichier->Append(ID_OUVRIR, "&Ouvrir archive..", "Ouvre l'archive");
        menuFichier->Append(ID_SAUVEGARDER, "&Sauvegarder archive..", "Sauvegarde l'archive");
        menuFichier->AppendSeparator();
        menuFichier->Append(ID_AJOUTER, "&Ajouter fichier..", "Ajouter un/des fichier(s) dans l'archive");
        menuFichier->Append(ID_SUPPRIMER, "&Supprimer fichier..", "Supprime un/des fichier(s) de l'archive");
        menuFichier->AppendSeparator();
        menuFichier->Append(ID_EXTRAIRE, "&Extraire fichier..", "Extrait un/des fichier(s) de l'archive");
        menuFichier->Append(ID_EXTRAIRE_TOUT, "&Extraire tout..", "Extrait tous les fichiers de l'archive");
        menuFichier->AppendSeparator();
        menuFichier->Append(ID_QUITTER, "&Quitter", "Quitter le programme");

        MenuBar->Append(menuFichier, "&Fichier");
    }
    
    // Menu Aide
    {
        wxMenu *menuAide = new wxMenu;
    
        menuAide->Append(ID_A_PROPOS, "A &Propos..", "Afficher des informations sur Archive(R)");
        
        MenuBar->Append(menuAide, "&Aide");
    }
    
    // Affecter le menu  l'application
    SetMenuBar(MenuBar);
}

// ---------------------------------------------------------------------------- INITIALISER LISTE
void GUI::initialiserListe()
{
    // crer le controle "liste des fichiers"
    m_liste = new wxListCtrl(this, 
                             ID_LISTE, 
                             wxDefaultPosition, 
                             wxDefaultSize, 
                             wxLC_REPORT,
                             wxDefaultValidator, 
                             "liste");
                             
    // Colonne Fichier
    {
        wxListItem colonne;
        
        m_liste->InsertColumn(0, "fichier", wxLIST_FORMAT_CENTRE, (int) (FENETRE_LARGEUR*0.68f));
        m_liste->InsertColumn(1, "compress", wxLIST_FORMAT_CENTRE, (int) (FENETRE_LARGEUR*0.13f));
        m_liste->InsertColumn(2, "original", wxLIST_FORMAT_CENTRE, (int) (FENETRE_LARGEUR*0.13f));
    }
}

// ---------------------------------------------------------------------------- CREER REPERTOIRE
void GUI::creerRepertoire(string &nomFichier)
{
	// Cre les repertoires des fichiers  extraire de l'archive..
 
    char directory[512];
	strcpy(directory, nomFichier.c_str());
	
	int iCurr = 0;	
	while ( iCurr<(int) strlen(directory) )
	{
		if ( directory[iCurr]=='\\' )
		{
			directory[iCurr] = 0;
			CreateDirectory( directory, NULL );
			directory[iCurr] = '\\';
		}
				
		iCurr++;
	}
}
    
// ---------------------------------------------------------------------------- ARCHIVE OUVRIR
string GUI::convertirEntierEnTaille(int taille)
{
    // Convertir un entier representation une taille en une chaine de caracteres 
    // qui sera affich dans l'outil !
    
    string tailleString;
    
    char nombre[16];
    
    if (taille<1024)
    {
        itoa(taille, nombre, 10);
        tailleString += nombre;
        tailleString += " octets";
    }
    else
    {
        itoa((taille/1024)+1, nombre, 10);
        tailleString += nombre;
        tailleString += " Ko";
    }
    
    return tailleString;
}
    
// ---------------------------------------------------------------------------- ARCHIVE OUVRIR
void GUI::archiveOuvrir(wxCommandEvent &event)
{
    wxFileDialog dialog(this,
                        "Ouvrir archive..",
                        "",
                        "",
                        "Fichiers Archive(R) (*.CDR)|*.CDR",
                        wxOPEN);
    
    // Afficher la boite de dialogue de selection de fichiers
    dialog.ShowModal();

    // Obtenir le nom du fichier de l'archive
    string nomFichier(dialog.GetFilename().c_str());
    if (nomFichier!="")
    {
        wxBusyCursor wait; // Place le curseur en attente. En quittant la methode il sera automatiquement remis a son etat normal..
    
        // Effacer tous les elements contenu dans notre controle Liste
        m_liste->DeleteAllItems();
        
        // Obtenir le nom du fichier de l'archive
        string nomFichier(dialog.GetDirectory());
        nomFichier += '\\';
        nomFichier += dialog.GetFilename().c_str();
        
        // Ouvrir l'archive
        ArchiveResultat res = m_archive.archiveOuvrir(nomFichier);
        if (res==ID_ARCHIVE_ERREUR_MAUVAISE_VERSION)
        {
            wxMessageDialog dialog(this, 
                                   "L'archive est invalide !",
                                   "Ouvrir archive..", 
                                   wxOK);

            // Afficher le message
            dialog.ShowModal();
            
            return;
        }
                
        // Obtenir le nombre de fichiers dans l'archive
        int nombreFichiers = m_archive.archiveObtenirNombreFichiers();
        
        // Pour chaques fichiers..
        for (int i=0; i<nombreFichiers; i++)
        {
            // Obtenir le nom du fichier courrant
            string nomFichier = m_archive.archiveObtenirNomFichierParIndice(i);
            
            // Inserer le fichier dans la liste
            m_liste->InsertItem(0, nomFichier.c_str());
            
            ArchiveElement element = m_archive.fichierObtenirElement(nomFichier);
            
            string tailleString;
            
            if (element.tailleCompressee==0 || element.typeCompression==ID_ARCHIVE_COMPRESSION_AUCUNE)
            {
                tailleString = "non";
            }
            else
            {
                tailleString = convertirEntierEnTaille(element.tailleCompressee);
            }    
                
            // Insere la taille compresse de l'element courrant dans la liste
            m_liste->SetItem(0, 1, tailleString.c_str());
            
            // Insere la taille originale de l'element courrant dans la liste
            tailleString = convertirEntierEnTaille(element.tailleOriginale);
            m_liste->SetItem(0, 2, tailleString.c_str());
        }
        
        wxMessageDialog dialog(this, 
                               "Operation effectue !",
                               "Ouvrir archive..", 
                               wxOK);

        // Afficher le message
        dialog.ShowModal();
    }
}

// ---------------------------------------------------------------------------- ARCHIVE SAUVEGARDER
void GUI::archiveSauvegarder(wxCommandEvent &event)
{
    // Controle que l'archive en cours d'edition n'est pas vide..
    if (m_liste->GetItemCount()==0)
    {
        wxMessageDialog dialog(this, 
                               "Aucun fichier dans l'archive !",
                               "Sauvegarder archive..", 
                               wxOK);

        // Afficher le message
        dialog.ShowModal();
        
        return;
    }
    
    wxFileDialog dialog(this,
                        "Sauvegarder archive..",
                        "",
                        "",
                        "Fichiers Archive(R) (*.CDR)|*.CDR",
                        wxSAVE);
    
    // Afficher la boite de dialogue de selection de fichiers
    dialog.ShowModal();
    
    // Obtenir le nom du fichier de l'archive
    string nomFichier(dialog.GetFilename().c_str());
    if (nomFichier!="")
    {
        wxBusyCursor wait; // place le curseur en attente. En quittant la methode il sera automatiquement remis a son etat normal..
    
        // Obtenir le nom du fichier courrant de l'archive
        string nomFichier(dialog.GetDirectory());
        nomFichier += '\\';
        nomFichier += dialog.GetFilename().c_str();
        
        // Sauvegarder l'archive
        m_archive.archiveSauvegarder(nomFichier);
        
        wxMessageDialog dialog(this, 
                               "Operation effectue !",
                               "Sauvegarder archive..", 
                               wxOK);

        // Afficher le message
        dialog.ShowModal();
    }
}

// ---------------------------------------------------------------------------- FICHIER AJOUTER
void GUI::fichierAjouter(wxCommandEvent &event)
{
    wxFileDialog dialog(this,
                        "Ajouter fichier..",
                        "",
                        "",
                        "Tous les fichiers (*.*)|*.*",
                        wxOPEN | wxMULTIPLE);
    
    // Afficher la boite de dialogue de selection de fichiers
    dialog.ShowModal();

    wxBusyCursor wait; // place le curseur en attente. En quittant la methode il sera automatiquement remis a son etat normal..

    // Obtenir la liste des fichiers  ajouter..
    wxArrayString nomFichiers;
    dialog.GetFilenames(nomFichiers);
    
    if (nomFichiers.GetCount()==0)
    {
        wxMessageDialog dialog(this, 
                               "Aucun fichier  ajouter !",
                               "Ajouter fichier..", 
                               wxOK);

        // Afficher le message
        dialog.ShowModal();
        
        return;
    }
    
    // Ajouter les fichiers  l'archive..
    for (int i=0; i<(int) nomFichiers.GetCount(); i++)
    {
        // Obtenir le nom du fichier courrant
        string nomFichier(dialog.GetDirectory());
        nomFichier += '\\';
        nomFichier += nomFichiers[i].c_str();
        
        // Ouvrir le fichier en lecture seule
        FILE *fichier = fopen(nomFichier.c_str(), "rb");
        
        // Connaitre la taille du fichier courrant
        fseek(fichier, 0, SEEK_END);
        int taille = ftell(fichier);
        fseek(fichier, 0, SEEK_SET);
        
        // Lire le contenu du fichier courrant
        char *contenu = new char[taille];
        fread(contenu, taille, 1, fichier);
        
        // Fermer le fichier courrant
        fclose(fichier);
        
        // Ajouter le fichier courrant dans l'Archive et controler si l'opration s'est bien passe
        if (m_archive.fichierAjouter(nomFichier, contenu, taille)==ID_ARCHIVE_ERREUR_FICHIER_DEJA_EXISTANT)
        {
            nomFichier += " est dja prsent dans l'archive !";
            
            wxMessageDialog dialog(this, 
                                   nomFichier.c_str(),
                                   "Ajouter fichier..", 
                                   wxOK);

            // Afficher le message
            dialog.ShowModal();
        }
        else
        {        
            // Obtenir l'element du fichier fraichement ajout
            ArchiveElement element = m_archive.fichierObtenirElement(nomFichier);
            
            // Ajouter le nom dans le controle Liste
            m_liste->InsertItem(0, nomFichier.c_str());

            string tailleString;
            
            if (element.tailleCompressee==0 || element.typeCompression==ID_ARCHIVE_COMPRESSION_AUCUNE)
            {
                tailleString = "non";
            }
            else
            {
                tailleString = convertirEntierEnTaille(element.tailleCompressee);
            }    
                
            // Insere la taille compresse de l'element courrant dans la liste
            m_liste->SetItem(0, 1, tailleString.c_str());
            
            // Insere la taille originale de l'element courrant dans la liste
            tailleString = convertirEntierEnTaille(element.tailleOriginale);
            m_liste->SetItem(0, 2, tailleString.c_str());
        }
        
        // Effacer le contenu original du fichier (qui est en mmoire)
        delete [] contenu;
    }
    
    {
        wxMessageDialog dialog(this, 
                               "Operation effectue !",
                               "Ajouter fichier..", 
                               wxOK);

        // Afficher le message
        dialog.ShowModal();
    }
}

// ---------------------------------------------------------------------------- FICHIER SUPPRIMER
void GUI::fichierSupprimer(wxCommandEvent &event)
{
    // Obtenir le 1er element du controle Liste qui est selectionn
    long i = m_liste->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
    
    // Si aucuns elements est selectionn..
    if (i==-1)
    {
        wxMessageDialog dialog(this, 
                               "Aucun fichier  supprimer !",
                               "Supprimer fichier..", 
                               wxOK);

        // Afficher le message
        dialog.ShowModal();
        
        return;
    }
    
    wxBusyCursor wait; // place le curseur en attente. En quittant la methode il sera automatiquement remis a son etat normal..
    
    // Tant qu'il y a des elements selectionns dans le controle Liste..
    while (i!=-1)
    {    
        // Obtenir le nom du fichier courrant
        string nomFichier(m_liste->GetItemText(i).c_str());
        
        // Supprimer le fichier courrant de l'archive
        m_archive.fichierSupprimer(nomFichier);
        
        // Effacer l'element courrant du controle Liste
        m_liste->DeleteItem(i);
        
        // Affecte  "i" le prochain element selectionn dans le controle Liste
        i = m_liste->GetNextItem(i-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
    }
    
    wxMessageDialog dialog(this, 
                           "Operation effectue !",
                           "Supprimer fichier..", 
                           wxOK);

    // Afficher le message
    dialog.ShowModal();
}

// ---------------------------------------------------------------------------- FICHIER EXTRAIRE
void GUI::fichierExtraire(wxCommandEvent &event)
{
    // Obtenir le 1er element du controle Liste qui est selectionn
    long i = m_liste->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
    
    // Si aucuns elements est selectionn..
    if (i==-1)
    {
        wxMessageDialog dialog(this, 
                               "Aucun fichier  extraire !",
                               "Extraire fichier..", 
                               wxOK);

        // Afficher le message
        dialog.ShowModal();
        
        return;
    }
    
    wxBusyCursor wait; // place le curseur en attente. En quittant la methode il sera automatiquement remis a son etat normal..
    
    // Tant qu'il y a des elements selectionns dans le controle Liste..
    while (i!=-1)
    {    
        // Obtenir le nom du fichier courrant
        string nomFichier(m_liste->GetItemText(i).c_str());
        
        // Creer le repertoire destination s'il n'existe pas dj !
        creerRepertoire(nomFichier);
        
        // Ouvrir le fichier en ecriture binaire
        FILE *fichier = fopen(nomFichier.c_str(), "wb");
        
        char *pointeur = NULL;
        int taille = 0;
        
        // Decompresser le contenu du fichier en mmoire
        // "pointeur" est automatiquement allou par la mthode "fichierDecompresser"
        m_archive.fichierDecompresser(nomFichier, pointeur, taille);
        
        // Ecrire le contenu decompress du fichier !
        fwrite(pointeur, taille, 1, fichier);
        
        // Fermer le fichier
        fclose(fichier);
        
        // Effacer le contenu decompress
        delete [] pointeur;
        
        // Affecte  "i" le prochain element selectionn dans le controle Liste
        i = m_liste->GetNextItem(i, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
    }
    
    wxMessageDialog dialog(this, 
                           "Operation effectue !",
                           "Extraire fichier..", 
                           wxOK);

    // Afficher le message
    dialog.ShowModal();
}

// ---------------------------------------------------------------------------- FICHIER EXTRAIRE TOUT
void GUI::fichierExtraireTout(wxCommandEvent &event)
{
    // Si aucuns elements n'est selectionn dans le controle Liste..
    if (m_liste->GetItemCount()==0)
    {
        wxMessageDialog dialog(this, 
                               "Aucun fichier  extraire !",
                               "Extraire tout..", 
                               wxOK);

        // Afficher le message
        dialog.ShowModal();
        
        return;
    }
    
    wxBusyCursor wait; // place le curseur en attente. En quittant la methode il sera automatiquement remis a son etat normal..
    
    // Obtenir le 1er element du controle Liste (qu'importe qu'il soit selectionn ou non)
    long i = m_liste->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_DONTCARE);
    
    // Tant qu'il y a des elements dans le controle Liste..
    while (i!=-1)
    {    
        // Obtenir le nom du fichier courrant
        string nomFichier(m_liste->GetItemText(i).c_str());
        
        // Creer le repertoire destination s'il n'existe pas dj !
        creerRepertoire(nomFichier);
        
        // Ouvrir le fichier en ecriture binaire
        FILE *fichier = fopen(nomFichier.c_str(), "wb");
        
        char *pointeur = NULL;
        int taille = 0;
        
        // Decompresser le contenu du fichier en mmoire
        // "pointeur" est automatiquement allou par la mthode "fichierDecompresser"
        m_archive.fichierDecompresser(nomFichier, pointeur, taille);
        
        // Ecrire le contenu decompress du fichier !
        fwrite(pointeur, taille, 1, fichier);
        
        // Fermer le fichier
        fclose(fichier);
        
        // Effacer le contenu decompress
        delete [] pointeur;
        
        // Affecte  "i" le prochain element selectionn dans le controle Liste
        i = m_liste->GetNextItem(i, wxLIST_NEXT_ALL, wxLIST_STATE_DONTCARE);
    }
    
    wxMessageDialog dialog(this, 
                           "Operation effectue !",
                           "Extraire tout..", 
                           wxOK);

    // Afficher le message
    dialog.ShowModal();
}

// ---------------------------------------------------------------------------- ARCHIVEUR A PROPOS
void GUI::archiveurAPropos(wxCommandEvent &event)
{
    wxMessageDialog dialog(this, 
                           "Archive(R)\n\n"
                           "Outil d'archivage permettant la manipulation d'archive au format CDR,\n"
                           "entierement realis avec Dev-C++ et wxWindows.\n\nhttp://www.posse-press.com",
                           "A Propos", 
                           wxOK);

    // Afficher le message
    dialog.ShowModal();
}

// ---------------------------------------------------------------------------- ARCHIVEUR QUITTER
void GUI::archiveurQuitter(wxCommandEvent &event)
{
    wxMessageDialog dialog(this, 
                           "Etes-vous sur ?",
                           "Quitter", 
                           wxYES | wxNO);

    // Afficher le message et teste si l'utilisateur a cliqu sur Oui    
    if (dialog.ShowModal()==wxID_YES)
    {    
        // Oui, alors quitter l'application !
        Close(TRUE);
    }
}
